/* 
 * Copyright 2012 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.model;

import java.util.Locale;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;

/* Represents the path of an associated resource */
public class ContactPath implements Parcelable, IContactable
{
	public enum Type
	{
		TAM, FAX, IMAGE, OTHER;

		/**
		 * Gets the text resource.
		 * 
		 * @param context
		 *            the context
		 * 
		 * @return the text resource
		 */
		public String getTextResource(Context context)
		{
			int id = context.getResources()
					.getIdentifier("contact_path_" + toString()
							.toLowerCase(Locale.US), "string",
							context.getApplicationContext().getPackageName());
			
			if (id == 0) return "";
			return context.getString(id);
		}
		
		/**
		 * Gets the path type for key.
		 * 
		 * @param key
		 *            the key
		 * 
		 * @return the path type for key
		 */
		public static Type getFromKey(String key)
		{
			for (Type type : Type.values())
			{
				if (key.toLowerCase(Locale.US).equals(type.toString()
						.toLowerCase(Locale.US)))
					return type;
			}
			// default
			return OTHER;
		}
	};

	protected Type mType = Type.OTHER;
	protected String mPath = "";

	/**
	 * Instantiates a new path.
	 */
	public ContactPath()
	{
		super();
	}

	/**
	 * Instantiates a new path.
	 */
	public ContactPath(String path, Type type)
	{
		super();
		setPath(path);
		setType(type);
	}
	
	public Type getType()
	{
		return mType;
	}

	/**
	 * Instantiates a new path.
	 * 
	 * @param in
	 *            the in
	 */
	public ContactPath(Parcel in)
	{
		mType = Type.values()[in.readInt()];
		mPath = in.readString();
	}

	/**
	 * Sets the type.
	 * 
	 * @param type
	 *            the new type
	 */
	public void setType(Type type)
	{
		mType = type;
	}

	/**
	 * Gets the path address.
	 * 
	 * @return the path address
	 */
	public String getPath()
	{
		return mPath;
	}

	/**
	 * Sets the path address.
	 * 
	 * @param path
	 *            the new path
	 */
	public void setPath(String path)
	{
		mPath = (path == null) ? "" : path;
	}

	public int describeContents()
	{
		return 0;
	}

	public void writeToParcel(Parcel out, int flags)
	{
		out.writeInt(mType.ordinal());
		out.writeString(mPath);
	}

	public static final Parcelable.Creator<ContactPath> CREATOR =
			new Parcelable.Creator<ContactPath>()
	{
		public ContactPath createFromParcel(Parcel in)
		{
			return new ContactPath(in);
		}

		public ContactPath[] newArray(int size)
		{
			return new ContactPath[size];
		}
	};

	public String getAddressDisplay()
	{
		return "";
	}

	public String getTypeDisplay(Context context)
	{
		return mType.getTextResource(context);
	}

	public String getAddress()
	{
		return getPath();
	}
}
